/*
 * $Id: Sql2Dot.java,v 1.3 2001/05/17 17:10:11 rdale Exp $
 *
 */

package net.jark.utils.db;

import java.io.*;
import java.util.*;

/**
 *
 * The database is a HashMap contain tables and a LinkedList for ordering
 *   "table name" : HashMap table
 *   LIST : LinkedList
 *
 * The table is a HashMap containing fields and a LinkedList for ordering
 *   "fieldName" : Integer fieldNumber
 *   EXCEPT in the case of references where
 *     "fieldName" : HashMap
 *   LIST : LinkedList
 *
 * The field HashMap contains
 *   "fieldNumber" : Integer fieldNumber
 *   "rTable"      : String reference table name
 *   "rField"      : String reference field name
 *
 * @author Robert Dale
 */
     

public class Sql2Dot
	{
	private HashMap db;
	private final String LIST = "__LIST__";

	Sql2Dot()
		{
		db = new HashMap();
		db.put(LIST, new LinkedList());
		}

	private void debug(String string)
		{
		System.err.println("debug: " + string);
		}

	private boolean isComment(String string)
		{
		String str = string.trim();
		if (str.startsWith("/*"))
			return true;
		if (str.startsWith("*"))
			return true;
		if (str.startsWith("//"))
			return true;
		if (str.startsWith("--"))
			return true;
		return false;
		}

	public void parseSql(Reader reader)
		{
		BufferedReader buff = new BufferedReader(reader);
		String line;
		try
			{
		while ((line = buff.readLine()) != null)
			{
			String tableName;
			if (isComment(line))
				continue;
			// find a table
			if (line.indexOf("create table") >= 0)
				{
				// do the dirty
				// find table name
				StringTokenizer tn = new StringTokenizer(line);
				tn.nextToken(); // create
				tn.nextToken(); // table
				tableName = tn.nextToken().trim(); // ahh, the table name!

				debug("Found table: " + tableName);

				// insert a new table in the database
				db.put(tableName,new HashMap());
				((LinkedList) db.get(LIST)).add(tableName);
				debug("Insert table: " + tableName);

				// create a list to order fields in the table
				((HashMap) db.get(tableName)).put(LIST, new LinkedList());
				
				// next token is either the beginning of the table
				// or a comment.  if it's not a comment, it's the
				// beginning of a table
				if (tn.hasMoreTokens())
					{
					String next = tn.nextToken();
					if (isComment(next))
						{
						// skip next line
						buff.readLine();
						}
					}
				else
					{
					// skip next line
					buff.readLine();
					}

				int fieldNumber = 0;
				// read in fields until end of table
				while (!(line = buff.readLine()).trim().startsWith(");"))
					{
					String fieldName = null;
					StringTokenizer fn = new StringTokenizer(line);

					// field name is always first
					if (fn.hasMoreTokens())
						{
						fieldName = fn.nextToken().trim();
						if (isComment(fieldName))
							continue;
						debug("Found field: " + fieldName);
						}
					else
						continue;

					fieldNumber++;

					// see if this is a reference
					String rTable = null; // reference table
					String rField = null; // reference field
					while (fn.hasMoreTokens())
						{
						String token = fn.nextToken();
						if (isComment(token))
							break;
						if (token.equals("references"))
							{
							rTable = fn.nextToken("(").trim();
							rField = fn.nextToken(")").substring(1).trim();
							debug("Found reference to: " +
											   rTable + "(" + rField + ")");
							}
						}

					// insert into table
					if (rTable != null)
						{
						HashMap ref = new HashMap();
						ref.put("rTable",rTable);
						ref.put("rField",rField);
						ref.put("fieldNumber",new Integer(fieldNumber));
						((HashMap) db.get(tableName)).put(fieldName, ref);
						debug("Insert field[" + fieldNumber + "]: " +
										   fieldName + " -> " +
										   rTable + "(" + rField + ")");
						}
					else
						{
						((HashMap) db.get(tableName)).put(fieldName, new Integer(fieldNumber));
						debug("Insert field[" + fieldNumber + "]:" +
										   fieldName);
						}
					((LinkedList) ((HashMap) db.get(tableName)).get(LIST)).add(fieldName);
					}
				}
			}
		buff.close();
			}
		catch (Exception e)
			{
			e.printStackTrace();
			}
		}

	public void toDot(Writer writer)
		throws java.io.IOException
		{
		String header = "digraph g {\n" +
			"size=\"10,7.5\"\n;" +
			"ratio=\"auto\"\n;" +
			"page=\"8.5,11\";\n" +
			"graph [\n" +
			"rankdir = \"LR\"\n" +
			"];\n" +
			"node [" +
			"fontsize = \"16\"\n" +
			"shape = \"ellipse\"\n" +
			"];\n" +
			"edge [\n" +
			"];\n";
		StringBuffer records = new StringBuffer();
		StringBuffer references = new StringBuffer();

		// each reference must have a unique id
		int refNum = 0;

		// iterate through all tables in database
		Iterator dbIter = ((LinkedList) db.get(LIST)).listIterator(0);
		while (dbIter.hasNext())
			{
			String tableName = (String) dbIter.next();
			HashMap tableMap = (HashMap) db.get(tableName);
			debug(tableName);

			// add table to record
			records.append("\"" + tableName + "\" [\n");
			// begin fields
			records.append("label = \"<f0> " + tableName);


			// iterate through all fields in table
			Iterator tableIter = ((LinkedList) tableMap.get(LIST)).listIterator(0);
			while (tableIter.hasNext())
				{
				HashMap map = null;
				Integer fieldNumber = null;
				String rTable = null;
				String rField = null;

				String fieldName = (String) tableIter.next();
				Object objHolder = tableMap.get(fieldName);

				// if field is a reference, iterate through its hashmap
				if (objHolder instanceof Integer)
					{
					fieldNumber = (Integer) objHolder;
					debug("  " + fieldNumber + "  " + fieldName);
					}
				else
					{
					refNum++;
					map = (HashMap) objHolder;
					fieldNumber = (Integer) map.get("fieldNumber");
					rTable = (String) map.get("rTable");
					rField = (String) map.get("rField");
					Integer intField = (Integer) ((HashMap) db.get(rTable)).get(rField);

					// add references
					references.append("\"" + tableName + "\":f" + fieldNumber +
									  " -> \"" + rTable + "\":f" + intField +
									  " [\n");
					references.append("id = " + refNum + "\n");
					references.append("];\n");
					debug("  " + fieldNumber + "  " + fieldName +
						  " references " + rTable +
						  "(" + rField + ")");
					}

				// add fields to record
				records.append("| <f" + fieldNumber + "> " + fieldName);

				

				}
			records.append("\"\n" +
						   "shape = \"record\"\n" +
						   "];\n");
			}
		// debug(header + records + references + "}");
		Writer righter = new PrintWriter(writer);
		righter.write(header + records + references + "}");
		righter.flush();
		righter.close();
		}

	private static void showHelp()
		{
		System.out.print("\n" +
						 "Usage: net.jark.utils.db.Sql2Dot <filename[s]>\n" +
						 "\n" +
						 "  - filename[s]:  one or more sql files\n" +
						 "  - output is in the same directory as filename with .dot extension\n" +
						 "\n");
		}

	public static void main(String[] args)
		{

		if (args.length < 1)
			{
			showHelp();
			System.exit(1);
			}

		String filename;
		for (int i = 0; i < args.length; i++)
			{
			filename = args[i];

			try
				{
				Reader reader = new FileReader(filename);
				Sql2Dot parser = new Sql2Dot();
				parser.parseSql(reader);

				Writer writer = new FileWriter(filename + ".dot");
				parser.toDot(writer);
				}
			catch (java.io.FileNotFoundException e)
				{
				System.err.println("File not found, skipping: " + filename);
				}
			catch (Exception e)
				{
				e.printStackTrace();
				}
			}
		}
	}

